<#
   The script adds paths, given as parameters, to the Microsoft Defender folder exclusion list,
   unless they are already excluded.
#>

#Requires -RunAsAdministrator

[Console]::OutputEncoding = [System.Text.Encoding]::UTF8

if ($args.Count -eq 0) {
  Write-Host "usage: $PSCommandPath path [path ...]"
  exit 1
}

try {
  Import-Module Defender

  # expands paths in the exclusion list
  function Expand-Excluded ([string[]] $exclusions) {
    $result = @()
    foreach ($exclusion in $exclusions) {
      try {
        $expanded = [System.Environment]::ExpandEnvironmentVariables($exclusion)
        $resolvedPaths = Resolve-Path -Path $expanded -ErrorAction Stop
        foreach ($resolved in $resolvedPaths) {
          $result += $resolved.ProviderPath
        }
      } catch [System.Management.Automation.ItemNotFoundException] {
      } catch [System.Management.Automation.DriveNotFoundException] {
      } catch [System.Management.Automation.WildcardPatternException] {
      } catch [System.UnauthorizedAccessException] {
      }
    }
    return $result
  }

  # returns `$true` when the given path is a sub-path of the given parent
  function Test-StartsWith ([string] $path, [string] $parent) {
    $pathNorm = $path.Trim('\') + '\'
    $parentNorm = $parent.Trim('\') + '\'
    return $pathNorm.Equals($parentNorm, [StringComparison]::OrdinalIgnoreCase) -or `
           $pathNorm.StartsWith($parentNorm, [StringComparison]::OrdinalIgnoreCase)
  }

  # returns `$true` when a path is already covered by the exclusion list
  function Test-Excluded ([string] $path, [string[]] $exclusions) {
    foreach ($exclusion in $exclusions) {
      if (Test-StartsWith $path $exclusion) {
        return $true
      }
    }
    return $false
  }

  $exclusions = (Get-MpPreference).ExclusionPath
  if (-not $exclusions) {
    $exclusions = @()
  } else {
    $exclusions = Expand-Excluded $exclusions
  }

  foreach ($path in $args) {
    if (-not (Test-Excluded $path $exclusions)) {
      $exclusions += $path
      Write-Host "added: $path"
    } else {
      Write-Host "skipped: $path"
    }
  }

  Set-MpPreference -ExclusionPath $exclusions
} catch {
  Write-Host "$($_.Exception.GetType()): $($_.Exception.Message)"
  Write-Host $_.ScriptStackTrace
  exit 1
}

# SIG # Begin signature block
# MIIuzAYJKoZIhvcNAQcCoIIuvTCCLrkCAQExDTALBglghkgBZQMEAgEweQYKKwYB
# BAGCNwIBBKBrMGkwNAYKKwYBBAGCNwIBHjAmAgMBAAAEEB/MO2BZSwhOtyTSxil+
# 81ECAQACAQACAQACAQACAQAwMTANBglghkgBZQMEAgEFAAQg9OPGiRg/9xyJzUyY
# SCqXYDVeviHTe8JjtES/dIowqHugghNYMIIF3zCCBMegAwIBAgIQTkDkN1Tt5owA
# AAAAUdOUfzANBgkqhkiG9w0BAQsFADCBvjELMAkGA1UEBhMCVVMxFjAUBgNVBAoT
# DUVudHJ1c3QsIEluYy4xKDAmBgNVBAsTH1NlZSB3d3cuZW50cnVzdC5uZXQvbGVn
# YWwtdGVybXMxOTA3BgNVBAsTMChjKSAyMDA5IEVudHJ1c3QsIEluYy4gLSBmb3Ig
# YXV0aG9yaXplZCB1c2Ugb25seTEyMDAGA1UEAxMpRW50cnVzdCBSb290IENlcnRp
# ZmljYXRpb24gQXV0aG9yaXR5IC0gRzIwHhcNMjEwNTA3MTU0MzQ1WhcNMzAxMTA3
# MTYxMzQ1WjBpMQswCQYDVQQGEwJVUzEWMBQGA1UECgwNRW50cnVzdCwgSW5jLjFC
# MEAGA1UEAww5RW50cnVzdCBDb2RlIFNpZ25pbmcgUm9vdCBDZXJ0aWZpY2F0aW9u
# IEF1dGhvcml0eSAtIENTQlIxMIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKC
# AgEAp4GP9xRFtmJD8tiu0yVeSE9Rv8V9n1AcNdHWfmEqlBltJ0akphpd91RRaoAi
# xqhmdU1Ug8leaBur9ltksK2tIL1U70ZrbQLnRa519o6KuTIui7h3HFJNeYhfpToY
# yVAslyctv9oAfWN/7zLsRodj25qfw1ohNnv5m9XKoG5yLPzh8Z5wTQhWFW+Qq/tI
# urnXwYJ4hWUuf7XJwOIUtzcRQQbiiuCo9uV+tngFAcNg7U8HQS4KE0njkJt/3b36
# rL9kUdFcm7T1XOdc/zubpaAa130JssK3/24cvMh95ukr/HKzFOlKVRKEnEQldR32
# KvBPpSA9aCXrYZd8D+W2PfOuw8ERvBuOzOBHMF5CAIZx41isBsplH3uUpktXZwx+
# Xq14Z1tV417rx9jsTG6Gy/Pc+J+HqnJYEg99pvj4Qjk7PCzkMk1JjODhAMI4oJz6
# hD5B3G5WrsYaW/RnaAUBzRu/roe8nVP2Lui2a+SZ3sVPh1io0mUeyB/Vcm7uWRxX
# OwlyndfKt5DGzXtFkpFCA0x9P8ryqrjCDobzEJ9GLqRmhmhaaBhwKTgRgGBrikOj
# c2zjs2s3/+adZwGSht8vSNH7UGDVXP4h0wFCY/7vcLQXwI+o7tPBS18S6v39Lg6H
# RGDjqfTCGKPj/c4MhCIN86d42pPz2zjPuS8zxv8HPF6+RdMCAwEAAaOCASswggEn
# MA4GA1UdDwEB/wQEAwIBhjASBgNVHRMBAf8ECDAGAQH/AgEBMB0GA1UdJQQWMBQG
# CCsGAQUFBwMDBggrBgEFBQcDCDA7BgNVHSAENDAyMDAGBFUdIAAwKDAmBggrBgEF
# BQcCARYaaHR0cDovL3d3dy5lbnRydXN0Lm5ldC9ycGEwMwYIKwYBBQUHAQEEJzAl
# MCMGCCsGAQUFBzABhhdodHRwOi8vb2NzcC5lbnRydXN0Lm5ldDAwBgNVHR8EKTAn
# MCWgI6Ahhh9odHRwOi8vY3JsLmVudHJ1c3QubmV0L2cyY2EuY3JsMB0GA1UdDgQW
# BBSCutY9l86fz3Hokjev/bO1aTVXzzAfBgNVHSMEGDAWgBRqciZ60B7vfec7aVHU
# bI2fkBJmqzANBgkqhkiG9w0BAQsFAAOCAQEAH15BBLaDcCRTLFVzHWU6wOy0ewSY
# Xlk4EwmkWZRCXlC/T2xuJSCQk1hADfUZtGLuJF7CAVgVAh0QCW+o1PuSfjc4Pi8U
# fY8dQzZks2YTXxTMpXH3WyFLxpe+3JX8cH0RHNMh3dAkOSnF/goapc97ee46b97c
# v+kR3RaDCNMsjX9NqBR5LwVhUjjrYPMUaH3LsoqtwJRc5CYOLIrdRsPO5FZRxVbj
# hbhNm0VyiwfxivtJuF/R8paBXWlSJPEII9LWIw/ri9d+i8GTa/rxYntY6VCbl24X
# iA3hxkOY14FhtoWdR+yxnq4/IDtDndiiHODUfAjCr3YG+GJmerb3+sivNTCCBoMw
# ggRroAMCAQICEDWvt3udNB9q/I+ERqsxNSswDQYJKoZIhvcNAQENBQAwaTELMAkG
# A1UEBhMCVVMxFjAUBgNVBAoMDUVudHJ1c3QsIEluYy4xQjBABgNVBAMMOUVudHJ1
# c3QgQ29kZSBTaWduaW5nIFJvb3QgQ2VydGlmaWNhdGlvbiBBdXRob3JpdHkgLSBD
# U0JSMTAeFw0yMTA1MDcxOTE5NTJaFw00MDEyMjkyMzU5MDBaMGMxCzAJBgNVBAYT
# AlVTMRYwFAYDVQQKEw1FbnRydXN0LCBJbmMuMTwwOgYDVQQDEzNFbnRydXN0IEV4
# dGVuZGVkIFZhbGlkYXRpb24gQ29kZSBTaWduaW5nIENBIC0gRVZDUzIwggIiMA0G
# CSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQC+vac5yaV97F1l8fQrqYfQ560axRo7
# GM7hoVGNcvrOWB9cuCRCD0bVMZfQSk3jmzDJupeonP5FNs8ngOd7uG7BJLffp3Tc
# hfKjJFhFzJSUOwLrxI18RvVcZoLjpMHTH6xuDFMjDtQ6+tpC8YNUXzUyVvK1eJtX
# AEgeqWFhJw5zA0O21nCS+9SFmjDGs+aaXkvvHSrYZqsWRv8L2A+miUoCUVdBPEE4
# TpfHUTJtZ45moV5NnzUir9Vqt39AX2g1zGn68QXw6oWm6jKFy8HByoNpRUkG3Als
# ukllGYz5tzcnjGSeNePl7OcHoJ2ocrxvhTosphZOPZzPCOaC9UR9KMC9ia1sL9wv
# eHkR1xxwS92dCExeLvqNvafdY/Z/8FIxhG462NlchUSeYwZp0IZYeImbh7tYHKQo
# bMb+aQqcHqwRYGpeyWllLu1DLWnxeLc7LTXyqk/iH+MBb5BGqtWoDQRXoLSs4229
# nRsogCdGx9qqZ5Xx0Yd7x8gl6YQMj4k20r4z4YXAM9WgPBmLrzjy5ZOAv8bDq3uT
# xD2due5FdsDUaG8wXjy0NvnXRULgEgaA26Uh/OcFeiiNtI5ge/fItUpHrBRml6Ba
# aXIxV2tThM1hunMHFXA7ewH8pz+MLT2HjPsI1+UvF4N+gqtPCfIp4X5Vd2WUBR1Z
# 1Ardk37jFF3iuwIDAQABo4IBKzCCAScwEgYDVR0TAQH/BAgwBgEB/wIBADAdBgNV
# HQ4EFgQUzolPglGqFaKEYsoxI2HSYfv4/ngwHwYDVR0jBBgwFoAUgrrWPZfOn89x
# 6JI3r/2ztWk1V88wMwYIKwYBBQUHAQEEJzAlMCMGCCsGAQUFBzABhhdodHRwOi8v
# b2NzcC5lbnRydXN0Lm5ldDAxBgNVHR8EKjAoMCagJKAihiBodHRwOi8vY3JsLmVu
# dHJ1c3QubmV0L2NzYnIxLmNybDAOBgNVHQ8BAf8EBAMCAYYwEwYDVR0lBAwwCgYI
# KwYBBQUHAwMwRAYDVR0gBD0wOzAwBgRVHSAAMCgwJgYIKwYBBQUHAgEWGmh0dHA6
# Ly93d3cuZW50cnVzdC5uZXQvcnBhMAcGBWeBDAEDMA0GCSqGSIb3DQEBDQUAA4IC
# AQA+AFS4KvOPZq9hFsRYk2T0QYtkVY0bNTOhm5HYq0bKq1+8vn5w5NLXB6iWB9eG
# 0VcKCNDQeE34Kt+yBPOa4dd88MEAesFAod+KjLfLtB0BYfKYmqbduIMFyqksFtyc
# LQ7+p5fkUKmXlcFLKoxR48QC4Gt8NU1TvEuUE4tTGBZepgqnuDTBbLHrLOQxfQws
# xnzhpFboLXbVshN16oMArgB3xm+pPe5jFIFQFvxbTxIZDlsPjepLzgZMiUuuIMyO
# R6Z11mXuLzDoXTSPH4JNXEKm8hRMUCCcCaJ0JFw52IkyhTyvjOVqnuYEOqUT/6od
# zUdLLgIFtGqP64VPge8K232fKY+lwj9SOFJBlTu8PltUMEIjCWPeUI2JNUX6q7gP
# j6Kte3oRk/GPKIR7aHlHauhkKU0f9B0vbR7IlIY801qZemt8qzX3KzlLT7k/FpLv
# KYyNq6wBGsxxRDnLQD4gEs4IcONH/tyA1wgA0Qtq6iG1eWYX6WqqVt3NFXixA8AU
# rT5HGHXtClNLMpPN/4CxkNYT5eRBeCLtQYLlDt+wzpUhReaLAdMZ9/QrVazZVnNj
# EOC4oG3LVJaYoJkhUQaqpgw6K2PLedpDudISzN6PVXrpCfPimtMlwXs2ktrg5VJn
# zePmVAqlndRzo9MvXQZkhQN3f2DYwrx0yZWRaLe3fmEZdTCCBuowggTSoAMCAQIC
# EDGdnUgatvXgkrzF40/3PFswDQYJKoZIhvcNAQELBQAwYzELMAkGA1UEBhMCVVMx
# FjAUBgNVBAoTDUVudHJ1c3QsIEluYy4xPDA6BgNVBAMTM0VudHJ1c3QgRXh0ZW5k
# ZWQgVmFsaWRhdGlvbiBDb2RlIFNpZ25pbmcgQ0EgLSBFVkNTMjAeFw0yMjEwMTEx
# MjM2MzdaFw0yNTEwMTExMjM2MzZaMIGaMQswCQYDVQQGEwJDWjEOMAwGA1UEBxMF
# UHJhaGExEzARBgsrBgEEAYI3PAIBAxMCQ1oxGTAXBgNVBAoTEEpldEJyYWlucyBz
# LnIuby4xHTAbBgNVBA8TFFByaXZhdGUgT3JnYW5pemF0aW9uMREwDwYDVQQFEwgy
# NjUwMjI3NTEZMBcGA1UEAxMQSmV0QnJhaW5zIHMuci5vLjCCAiIwDQYJKoZIhvcN
# AQEBBQADggIPADCCAgoCggIBAMmZsBOO0sV6f+qwSfowL7CAaqooz6zzYDcPWA/6
# P1BlOeO9NqKEzd4IyZEI1lccipydIwdaO1YHa2kM7/5kertoyJ6ITuMjmcyxnqXI
# AhdCHLbqO/Vq6R+4fqokQ5FvPvsTZ6bbc8sznX3roST/zjlgYA5+HDR2GRZ4sy5d
# tkt/i/MANl3f3rRv1RO0qvy+dU7GOk0CWPfXFIHrj+sszSQ7zgM52bHAP70NlKfj
# VkKdX7U4Ytz3yb1gnfLRQyxYgM/MBKGDI6BrHlUfskryxjV5gH+aWFfBaXw8+NVl
# fYr8lUdlXlzlWmHqa7J792WNCLlE31NevEYaXA+0TbxJKT8bvXWehHxBpGY9Q15z
# WjnaGk1FguXxqmXKkRQfDgBiJ4sCtGW3iVGmOtMiEKmOURSZ/hoUQSrtMf8r/itK
# re6DHoxGbjA9yjzPXZbT1dJk6eOcKZaesYANyu05Kz7S/lRX83N4UbCQEJ4xpcCI
# G+eeE4d2BrapVFMZxobUKZaFtV+SAByQFEumEjEX5hEMyGzfODoa0KnFowf9Fu5u
# lz0dM3cV6/+riIgphUIlsNYgum7swNn6dp13+iyMuaOvPEL0kBlOPufcz7Lq18a6
# o5anOBJpW9jAMf0cCfJ+hwAOj3gcTMbiKCmEo7gEoQiZvnM4DZra7tf+Nblmv730
# RtPHAgMBAAGjggFgMIIBXDAMBgNVHRMBAf8EAjAAMB0GA1UdDgQWBBQKghd/k2G5
# FxhP/0MIUgmYqUN3xjAfBgNVHSMEGDAWgBTOiU+CUaoVooRiyjEjYdJh+/j+eDBn
# BggrBgEFBQcBAQRbMFkwIwYIKwYBBQUHMAGGF2h0dHA6Ly9vY3NwLmVudHJ1c3Qu
# bmV0MDIGCCsGAQUFBzAChiZodHRwOi8vYWlhLmVudHJ1c3QubmV0L2V2Y3MyLWNo
# YWluLnA3YzAxBgNVHR8EKjAoMCagJKAihiBodHRwOi8vY3JsLmVudHJ1c3QubmV0
# L2V2Y3MyLmNybDAOBgNVHQ8BAf8EBAMCB4AwEwYDVR0lBAwwCgYIKwYBBQUHAwMw
# SwYDVR0gBEQwQjA3BgpghkgBhvpsCgECMCkwJwYIKwYBBQUHAgEWG2h0dHBzOi8v
# d3d3LmVudHJ1c3QubmV0L3JwYTAHBgVngQwBAzANBgkqhkiG9w0BAQsFAAOCAgEA
# BLLwIeBU0HShBD5L1ZoZN+ZQIpMOIHkQajuxja1zSLwakCX3lA0V3MpxliDDRx3e
# y4YA3OcefN2zV6xUAVNOb6V8zB552SblNVxa4TIxzmSzKI2uifjkikt1ZogDEXsS
# Cflyak2rbBATAmNrEBFAcZBLUGDxExbK2HyZAtG+CR16jj9Qd3zDHmbSciIMlsBq
# pmgmP34/pcjr/LfjXNqa2r0Q+ISbhVgv5N4ESYdBUKh0SAMf5fcUAreP0IpTOO4v
# FkkZSoUkALhGfBQDluukYrIGUh7fjmNVAwTQ1UjLRb8VDQwsIhFMBJkzevbxkqZA
# 3O6JLdisMuRL6/CFkdnftPAeiBJbs2jRQzADDGylIdIMv8V6A/yymtg9kZH9eCNw
# JibhEhTPHsEJX5Unwk8vE7POUqCKoB7+ULkGURTrUtEBBYmShPcbjQH0l6pcb66J
# oCX78Cbzd/Zr9Fm6mLjjlNbZcyjBSGDyuZq6SpKLL90YYCXHNEFKJdFCtnxtnbM4
# ipy3TZi9Xhx5OSfLd3T7/WgFQYuLHacteeGVonYBGXZbQiFKbUophNfYFRF0N5ZE
# cc7Vy3Vm+JSAvX496v6GRR3/C2zLI9ffZBqYLFhMGo0qIIUNLuIBbhX6pzw3mrlj
# VIdeZGNRj3vWqhAp9A7IzS6x3t2RrmSrkV1an7k6Um0xghrMMIIayAIBATB3MGMx
# CzAJBgNVBAYTAlVTMRYwFAYDVQQKEw1FbnRydXN0LCBJbmMuMTwwOgYDVQQDEzNF
# bnRydXN0IEV4dGVuZGVkIFZhbGlkYXRpb24gQ29kZSBTaWduaW5nIENBIC0gRVZD
# UzICEDGdnUgatvXgkrzF40/3PFswCwYJYIZIAWUDBAIBoIGuMBkGCSqGSIb3DQEJ
# AzEMBgorBgEEAYI3AgEEMBwGCisGAQQBgjcCAQsxDjAMBgorBgEEAYI3AgEVMC8G
# CSqGSIb3DQEJBDEiBCAcdR8jR9bL9V3fqV5tBa/+1UUg1PuQ4DNM/0aeA9a25jBC
# BgorBgEEAYI3AgEMMTQwMqAwgC4AZABlAGYAZQBuAGQAZQByAC0AZQB4AGMAbAB1
# AHMAaQBvAG4AcwAuAHAAcwAxMA0GCSqGSIb3DQEBAQUABIICAFocIHHprDje8cVC
# mFj74BgwsoTqkNRouDVx3ix/abJdz8ysrV70+kXJrIsRqJ0HwTBdDYsRXV+dCBiI
# 2Q7bFBgo9lioCk3huX0JURLqa0r7Af5yNX1N3eEY1vNxg3rB5nK8VQjbYpHE30yU
# sePGc0FgqmuwReOR6Ycsdi8WRHOAQnHUEPjHvJ1bAEJn26Cbi/GJxa7SRSy4GTtS
# SbmIsIDcOyw3g0euA6CnFKV5IqI3DZBI3V8bwzN0rP/WV2dcbZYqL6oM9B84uLZh
# JgfTPYH/16fqlnm3e4gQx0Cz2RSqVSKn/Iol0044KxTkce+jj8Gm/xAdpkLOp2yt
# JatMn4Ld3sFiIsdTLRxdZW5zLUSqX6wSLrx8p78+UPi1rWXV6Jns7bkNzgNfJhmA
# GPPPFYgwtbumLWuvrcOJB3DYdiNMFSNUokc+5UmPEbOlynNv9weIQM9LpMlGWtvR
# Mz0ViyCUJKFzjm+JT8Aq0L0mdxtL4CmXQRkEJrGiM2AIDEOvpCBxDPGcQUBmSQCP
# EyxMUrRcU4t1jmi9BJYrINT9MJwTb14J0UYxFYT9A+lSME0gtISz6LnVQUphIWQ+
# dK43eYaN3d0cYwa4gx8hpA1YUc2ZNldEuk/RWow9UHu+Ungn8oFIC17s+d+CE5z9
# GvJM1iZFjq3s963H2BDcRf8GpR9VoYIXdzCCF3MGCisGAQQBgjcDAwExghdjMIIX
# XwYJKoZIhvcNAQcCoIIXUDCCF0wCAQMxDzANBglghkgBZQMEAgEFADB4BgsqhkiG
# 9w0BCRABBKBpBGcwZQIBAQYJYIZIAYb9bAcBMDEwDQYJYIZIAWUDBAIBBQAEILs1
# gqDtCo4s+uK2iQ1h9omXB/ePb5JtXaHrrW67lXRmAhEAhp2R0rlXzI3CyRqz4SoH
# ShgPMjAyNTA5MjQxODU0NTBaoIITOjCCBu0wggTVoAMCAQICEAqA7xhLjfEFgtHE
# dqeVdGgwDQYJKoZIhvcNAQELBQAwaTELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDkRp
# Z2lDZXJ0LCBJbmMuMUEwPwYDVQQDEzhEaWdpQ2VydCBUcnVzdGVkIEc0IFRpbWVT
# dGFtcGluZyBSU0E0MDk2IFNIQTI1NiAyMDI1IENBMTAeFw0yNTA2MDQwMDAwMDBa
# Fw0zNjA5MDMyMzU5NTlaMGMxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2Vy
# dCwgSW5jLjE7MDkGA1UEAxMyRGlnaUNlcnQgU0hBMjU2IFJTQTQwOTYgVGltZXN0
# YW1wIFJlc3BvbmRlciAyMDI1IDEwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIK
# AoICAQDQRqwtEsae0OquYFazK1e6b1H/hnAKAd/KN8wZQjBjMqiZ3xTWcfsLwOvR
# xUwXcGx8AUjni6bz52fGTfr6PHRNv6T7zsf1Y/E3IU8kgNkeECqVQ+3bzWYesFtk
# epErvUSbf+EIYLkrLKd6qJnuzK8Vcn0DvbDMemQFoxQ2Dsw4vEjoT1FpS54dNApZ
# fKY61HAldytxNM89PZXUP/5wWWURK+IfxiOg8W9lKMqzdIo7VA1R0V3Zp3DjjANw
# qAf4lEkTlCDQ0/fKJLKLkzGBTpx6EYevvOi7XOc4zyh1uSqgr6UnbksIcFJqLbkI
# XIPbcNmA98Oskkkrvt6lPAw/p4oDSRZreiwB7x9ykrjS6GS3NR39iTTFS+ENTqW8
# m6THuOmHHjQNC3zbJ6nJ6SXiLSvw4Smz8U07hqF+8CTXaETkVWz0dVVZw7knh1WZ
# XOLHgDvundrAtuvz0D3T+dYaNcwafsVCGZKUhQPL1naFKBy1p6llN3QgshRta6Eq
# 4B40h5avMcpi54wm0i2ePZD5pPIssoszQyF4//3DoK2O65Uck5Wggn8O2klETsJ7
# u8xEehGifgJYi+6I03UuT1j7FnrqVrOzaQoVJOeeStPeldYRNMmSF3voIgMFtNGh
# 86w3ISHNm0IaadCKCkUe2LnwJKa8TIlwCUNVwppwn4D3/Pt5pwIDAQABo4IBlTCC
# AZEwDAYDVR0TAQH/BAIwADAdBgNVHQ4EFgQU5Dv88jHt/f3X85FxYxlQQ89hjOgw
# HwYDVR0jBBgwFoAU729TSunkBnx6yuKQVvYv1Ensy04wDgYDVR0PAQH/BAQDAgeA
# MBYGA1UdJQEB/wQMMAoGCCsGAQUFBwMIMIGVBggrBgEFBQcBAQSBiDCBhTAkBggr
# BgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMF0GCCsGAQUFBzAChlFo
# dHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRUaW1l
# U3RhbXBpbmdSU0E0MDk2U0hBMjU2MjAyNUNBMS5jcnQwXwYDVR0fBFgwVjBUoFKg
# UIZOaHR0cDovL2NybDMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZEc0VGlt
# ZVN0YW1waW5nUlNBNDA5NlNIQTI1NjIwMjVDQTEuY3JsMCAGA1UdIAQZMBcwCAYG
# Z4EMAQQCMAsGCWCGSAGG/WwHATANBgkqhkiG9w0BAQsFAAOCAgEAZSqt8RwnBLmu
# YEHs0QhEnmNAciH45PYiT9s1i6UKtW+FERp8FgXRGQ/YAavXzWjZhY+hIfP2JkQ3
# 8U+wtJPBVBajYfrbIYG+Dui4I4PCvHpQuPqFgqp1PzC/ZRX4pvP/ciZmUnthfAEP
# 1HShTrY+2DE5qjzvZs7JIIgt0GCFD9ktx0LxxtRQ7vllKluHWiKk6FxRPyUPxAAY
# H2Vy1lNM4kzekd8oEARzFAWgeW3az2xejEWLNN4eKGxDJ8WDl/FQUSntbjZ80FU3
# i54tpx5F/0Kr15zW/mJAxZMVBrTE2oi0fcI8VMbtoRAmaaslNXdCG1+lqvP4FbrQ
# 6IwSBXkZagHLhFU9HCrG/syTRLLhAezu/3Lr00GrJzPQFnCEH1Y58678IgmfORBP
# C1JKkYaEt2OdDh4GmO0/5cHelAK2/gTlQJINqDr6JfwyYHXSd+V08X1JUPvB4ILf
# JdmL+66Gp3CSBXG6IwXMZUXBhtCyIaehr0XkBoDIGMUG1dUtwq1qmcwbdUfcSYCn
# +OwncVUXf53VJUNOaMWMts0VlRYxe5nK+At+DI96HAlXHAL5SlfYxJ7La54i71Mc
# VWRP66bW+yERNpbJCjyCYG2j+bdpxo/1Cy4uPcU3AWVPGrbn5PhDBf3Froguzzhk
# ++ami+r3Qrx5bIbY3TVzgiFI7Gq3zWcwgga0MIIEnKADAgECAhANx6xXBf8hmS5A
# QyIMOkmGMA0GCSqGSIb3DQEBCwUAMGIxCzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxE
# aWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xITAfBgNVBAMT
# GERpZ2lDZXJ0IFRydXN0ZWQgUm9vdCBHNDAeFw0yNTA1MDcwMDAwMDBaFw0zODAx
# MTQyMzU5NTlaMGkxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwgSW5j
# LjFBMD8GA1UEAxM4RGlnaUNlcnQgVHJ1c3RlZCBHNCBUaW1lU3RhbXBpbmcgUlNB
# NDA5NiBTSEEyNTYgMjAyNSBDQTEwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIK
# AoICAQC0eDHTCphBcr48RsAcrHXbo0ZodLRRF51NrY0NlLWZloMsVO1DahGPNRcy
# bEKq+RuwOnPhof6pvF4uGjwjqNjfEvUi6wuim5bap+0lgloM2zX4kftn5B1IpYzT
# qpyFQ/4Bt0mAxAHeHYNnQxqXmRinvuNgxVBdJkf77S2uPoCj7GH8BLuxBG5AvftB
# dsOECS1UkxBvMgEdgkFiDNYiOTx4OtiFcMSkqTtF2hfQz3zQSku2Ws3IfDReb6e3
# mmdglTcaarps0wjUjsZvkgFkriK9tUKJm/s80FiocSk1VYLZlDwFt+cVFBURJg6z
# MUjZa/zbCclF83bRVFLeGkuAhHiGPMvSGmhgaTzVyhYn4p0+8y9oHRaQT/aofEnS
# 5xLrfxnGpTXiUOeSLsJygoLPp66bkDX1ZlAeSpQl92QOMeRxykvq6gbylsXQskBB
# BnGy3tW/AMOMCZIVNSaz7BX8VtYGqLt9MmeOreGPRdtBx3yGOP+rx3rKWDEJlIqL
# XvJWnY0v5ydPpOjL6s36czwzsucuoKs7Yk/ehb//Wx+5kMqIMRvUBDx6z1ev+7ps
# NOdgJMoiwOrUG2ZdSoQbU2rMkpLiQ6bGRinZbI4OLu9BMIFm1UUl9VnePs6BaaeE
# WvjJSjNm2qA+sdFUeEY0qVjPKOWug/G6X5uAiynM7Bu2ayBjUwIDAQABo4IBXTCC
# AVkwEgYDVR0TAQH/BAgwBgEB/wIBADAdBgNVHQ4EFgQU729TSunkBnx6yuKQVvYv
# 1Ensy04wHwYDVR0jBBgwFoAU7NfjgtJxXWRM3y5nP+e6mK4cD08wDgYDVR0PAQH/
# BAQDAgGGMBMGA1UdJQQMMAoGCCsGAQUFBwMIMHcGCCsGAQUFBwEBBGswaTAkBggr
# BgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMEEGCCsGAQUFBzAChjVo
# dHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkUm9vdEc0
# LmNydDBDBgNVHR8EPDA6MDigNqA0hjJodHRwOi8vY3JsMy5kaWdpY2VydC5jb20v
# RGlnaUNlcnRUcnVzdGVkUm9vdEc0LmNybDAgBgNVHSAEGTAXMAgGBmeBDAEEAjAL
# BglghkgBhv1sBwEwDQYJKoZIhvcNAQELBQADggIBABfO+xaAHP4HPRF2cTC9vgvI
# tTSmf83Qh8WIGjB/T8ObXAZz8OjuhUxjaaFdleMM0lBryPTQM2qEJPe36zwbSI/m
# S83afsl3YTj+IQhQE7jU/kXjjytJgnn0hvrV6hqWGd3rLAUt6vJy9lMDPjTLxLgX
# f9r5nWMQwr8Myb9rEVKChHyfpzee5kH0F8HABBgr0UdqirZ7bowe9Vj2AIMD8liy
# rukZ2iA/wdG2th9y1IsA0QF8dTXqvcnTmpfeQh35k5zOCPmSNq1UH410ANVko43+
# Cdmu4y81hjajV/gxdEkMx1NKU4uHQcKfZxAvBAKqMVuqte69M9J6A47OvgRaPs+2
# ykgcGV00TYr2Lr3ty9qIijanrUR3anzEwlvzZiiyfTPjLbnFRsjsYg39OlV8cipD
# oq7+qNNjqFzeGxcytL5TTLL4ZaoBdqbhOhZ3ZRDUphPvSRmMThi0vw9vODRzW6Ax
# nJll38F0cuJG7uEBYTptMSbhdhGQDpOXgpIUsWTjd6xpR6oaQf/DJbg3s6KCLPAl
# Z66RzIg9sC+NJpud/v4+7RWsWCiKi9EOLLHfMR2ZyJ/+xhCx9yHbxtl5TPau1j/1
# MIDpMPx0LckTetiSuEtQvLsNz3Qbp7wGWqbIiOWCnb5WqxL3/BAPvIXKUjPSxyZs
# q8WhbaM2tszWkPZPubdcMIIFjTCCBHWgAwIBAgIQDpsYjvnQLefv21DiCEAYWjAN
# BgkqhkiG9w0BAQwFADBlMQswCQYDVQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQg
# SW5jMRkwFwYDVQQLExB3d3cuZGlnaWNlcnQuY29tMSQwIgYDVQQDExtEaWdpQ2Vy
# dCBBc3N1cmVkIElEIFJvb3QgQ0EwHhcNMjIwODAxMDAwMDAwWhcNMzExMTA5MjM1
# OTU5WjBiMQswCQYDVQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYD
# VQQLExB3d3cuZGlnaWNlcnQuY29tMSEwHwYDVQQDExhEaWdpQ2VydCBUcnVzdGVk
# IFJvb3QgRzQwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQC/5pBzaN67
# 5F1KPDAiMGkz7MKnJS7JIT3yithZwuEppz1Yq3aaza57G4QNxDAf8xukOBbrVsaX
# bR2rsnnyyhHS5F/WBTxSD1Ifxp4VpX6+n6lXFllVcq9ok3DCsrp1mWpzMpTREEQQ
# Lt+C8weE5nQ7bXHiLQwb7iDVySAdYyktzuxeTsiT+CFhmzTrBcZe7FsavOvJz82s
# NEBfsXpm7nfISKhmV1efVFiODCu3T6cw2Vbuyntd463JT17lNecxy9qTXtyOj4Da
# tpGYQJB5w3jHtrHEtWoYOAMQjdjUN6QuBX2I9YI+EJFwq1WCQTLX2wRzKm6RAXwh
# TNS8rhsDdV14Ztk6MUSaM0C/CNdaSaTC5qmgZ92kJ7yhTzm1EVgX9yRcRo9k98Fp
# iHaYdj1ZXUJ2h4mXaXpI8OCiEhtmmnTK3kse5w5jrubU75KSOp493ADkRSWJtppE
# GSt+wJS00mFt6zPZxd9LBADMfRyVw4/3IbKyEbe7f/LVjHAsQWCqsWMYRJUadmJ+
# 9oCw++hkpjPRiQfhvbfmQ6QYuKZ3AeEPlAwhHbJUKSWJbOUOUlFHdL4mrLZBdd56
# rF+NP8m800ERElvlEFDrMcXKchYiCd98THU/Y+whX8QgUWtvsauGi0/C1kVfnSD8
# oR7FwI+isX4KJpn15GkvmB0t9dmpsh3lGwIDAQABo4IBOjCCATYwDwYDVR0TAQH/
# BAUwAwEB/zAdBgNVHQ4EFgQU7NfjgtJxXWRM3y5nP+e6mK4cD08wHwYDVR0jBBgw
# FoAUReuir/SSy4IxLVGLp6chnfNtyA8wDgYDVR0PAQH/BAQDAgGGMHkGCCsGAQUF
# BwEBBG0wazAkBggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMEMG
# CCsGAQUFBzAChjdodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRB
# c3N1cmVkSURSb290Q0EuY3J0MEUGA1UdHwQ+MDwwOqA4oDaGNGh0dHA6Ly9jcmwz
# LmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydEFzc3VyZWRJRFJvb3RDQS5jcmwwEQYDVR0g
# BAowCDAGBgRVHSAAMA0GCSqGSIb3DQEBDAUAA4IBAQBwoL9DXFXnOF+go3QbPbYW
# 1/e/Vwe9mqyhhyzshV6pGrsi+IcaaVQi7aSId229GhT0E0p6Ly23OO/0/4C5+KH3
# 8nLeJLxSA8hO0Cre+i1Wz/n096wwepqLsl7Uz9FDRJtDIeuWcqFItJnLnU+nBgMT
# dydE1Od/6Fmo8L8vC6bp8jQ87PcDx4eo0kxAGTVGamlUsLihVo7spNU96LHc/RzY
# 9HdaXFSMb++hUD38dglohJ9vytsgjTVgHAIDyyCwrFigDkBjxZgiwbJZ9VVrzyer
# bHbObyMt9H5xaiNrIv8SuFQtJ37YOtnwtoeW/VvRXKwYw02fc7cBqZ9Xql4o4rmU
# MYIDfDCCA3gCAQEwfTBpMQswCQYDVQQGEwJVUzEXMBUGA1UEChMORGlnaUNlcnQs
# IEluYy4xQTA/BgNVBAMTOERpZ2lDZXJ0IFRydXN0ZWQgRzQgVGltZVN0YW1waW5n
# IFJTQTQwOTYgU0hBMjU2IDIwMjUgQ0ExAhAKgO8YS43xBYLRxHanlXRoMA0GCWCG
# SAFlAwQCAQUAoIHRMBoGCSqGSIb3DQEJAzENBgsqhkiG9w0BCRABBDAcBgkqhkiG
# 9w0BCQUxDxcNMjUwOTI0MTg1NDUwWjArBgsqhkiG9w0BCRACDDEcMBowGDAWBBTd
# YjCshgotMGvaOLFoeVIwB/tBfjAvBgkqhkiG9w0BCQQxIgQgBEPClfTjbjlPFscO
# f7LGWNle22zxLZgpijwH12V9+DcwNwYLKoZIhvcNAQkQAi8xKDAmMCQwIgQgSqA/
# oizXXITFXJOPgo5na5yuyrM/420mmqM08UYRCjMwDQYJKoZIhvcNAQEBBQAEggIA
# MSnWoUqKs3tdTNa3hDJLZ5TGJshBFDFTGaGH6Ayp8YDervDDMbe9Nd12ld4SUSpP
# 37JnZ5LimtsC+CI3M4/s78neD4pLwUD1OtAu3qCJxr6lgH74HI30pIpHOXKVMVUi
# /U3oy533vnnvlJXRhsmoMDwgjShjXXLKvaRCWEtG+GpWcBOzplNejEVXEj2Un3YJ
# nXw5+aIMj2A3wpGv4Raw0N2A7MmUR6Ql0L+EW+ugOwX3La1EU6GYOrkLykSLP4dE
# Ao8Kw6Sla82ZLgGIDTndw8YGbVFl7EcPDlmW9YsjFnggzF5aGO1iBKMh9h2mZbPN
# sIHLtdAZrMDxrEEa7shtjYmpH/v6R3Sn+4gCResDASTRSeh/w13C2YSTDuiFwiNb
# SiAeUGRoYAhHlk1dO7NoYWA2FupRoCs9YVlVfDAkIntutyr+di9enqvFjZGvhfWA
# 2ZkNAQEi94/tbyZZoaL4XyQXX3MGZjs6Q9FuJb3TBd25g1AczemqzqL8IVQ9vDN0
# 9NIhkO1QLM23ozN08tJH60F7j4CgAI/c1KHavPg9WEb8XsEqKDlrx0/+nZ4YA7b8
# f5pDqNkCwjeGBq/NSEMohYg3iVJE8MkD+ACokTu6JazMP1ErYhq9aDWTkuoT5Kk0
# XuPc+ZrE3Ir7ifr71I5Ue9MNGVCufMynmnLMADDFwF0=
# SIG # End signature block
