// based on https://github.com/vuejs/language-tools/blob/v3.0.0-alpha.8/extensions/vscode/rolldown.config.ts

import * as fs from 'node:fs';
import * as path from 'node:path';
import {defineConfig, type RolldownOptions} from 'rolldown';

const resolve = (...paths: string[]) => path.resolve(__dirname, ...paths);

const packageRelativePath = "index.js".replace(".js", "")
const nodeModulesPath = resolve('node_modules')
const ownPackageJson = require(resolve('./package.json'))
const packageName = ownPackageJson.name
const theirPackageJson = require(path.resolve(nodeModulesPath, packageName, 'package.json'))

if (ownPackageJson.version !== theirPackageJson.version) {
  throw new Error(`Make sure that the version in package.json matches the version of official server distribution (${theirPackageJson.version})`)
}

const config: RolldownOptions = {
  input: {
    [packageRelativePath]: path.resolve(nodeModulesPath, packageName, packageRelativePath),
  },
  output: {
    format: 'cjs',
    sourcemap: false,
    dir: resolve(),
  },
  define: {
    'process.env.NODE_ENV': '"production"',
  },
  plugins: [
    {
      name: 'redirect',
      writeBundle(options, bundle) {
        fs.rmSync(nodeModulesPath, {recursive: true})
        console.log("Done.")
      },
    },
    {
      name: 'umd2esm',
      resolveId: {
        filter: {
          id: /^(vscode-.*-languageservice|vscode-languageserver-types|jsonc-parser)$/,
        },
        handler(source, importer) {
          const pathUmdMay = require.resolve(source, {paths: [importer!]});
          // Call twice the replace is to solve the problem of the path in Windows
          const pathEsm = pathUmdMay.replace('/umd/', '/esm/').replace('\\umd\\', '\\esm\\');
          return {id: pathEsm};
        },
      },
    },
  ],
};

export default defineConfig(config);
