package com.intellij.ide.starter.project

import com.fasterxml.jackson.annotation.JsonProperty
import com.fasterxml.jackson.databind.ObjectMapper
import com.intellij.ide.starter.ide.IDETestContext
import java.net.URLEncoder
import java.nio.charset.StandardCharsets
import java.nio.file.Path
import kotlin.time.Duration
import kotlin.time.Duration.Companion.minutes

/**
 * Project generated by KMP Wizard.
 */

data class KmpWizardProjectInfo(
  val projectName: String,
  val kmpProjectSpec: KmpProjectSpec,
  override val isReusable: Boolean = false,
  override val downloadTimeout: Duration = 10.minutes,
  override val configureProjectBeforeUse: (IDETestContext) -> Unit = {},
  private val description: String = "",
) : ProjectInfoSpec {

  fun buildKmpProjectUrl(projectName: String, projectSpec: KmpProjectSpec): String {
    val projectSpecAsJson = ObjectMapper().writer().writeValueAsString(projectSpec)

    val baseUrl = "https://kmp.jetbrains.com/generateKmtProject"
    val params = linkedMapOf(
      "name" to projectName,
      "id" to "org.example.project",
      ":r2::" to "Compose",
      "spec" to projectSpecAsJson
    )

    val query = params.entries.joinToString("&") { (key, value) ->
      "${URLEncoder.encode(key, StandardCharsets.UTF_8)}=${URLEncoder.encode(value, StandardCharsets.UTF_8)}"
    }

    return "$baseUrl?$query"
  }


  override fun downloadAndUnpackProject(): Path = RemoteArchiveProjectInfo(buildKmpProjectUrl(projectName, kmpProjectSpec)).downloadAndUnpackProject()

  override fun getDescription(): String {
    return description
  }

  // DTOs for KMP Wizard spec
  data class KmpProjectSpec(
    @get:JsonProperty("template_id") val templateId: TemplateId = TemplateId.Kmt,
    @get:JsonProperty("targets") val targets: Map<Target.Id, Map<Option.Id, Set<Variant.Id>>>
  )

  enum class TemplateId {
    @JsonProperty("kmt-example") KmtExample,
    @JsonProperty("kmt") Kmt,
  }

  data class Target(@JsonProperty("options") val options: Map<Option.Id, Option>) {
    enum class Id {
      @JsonProperty("android") Android,
      @JsonProperty("ios") iOS,
      @JsonProperty("web") Web,
      @JsonProperty("desktop") Desktop,
      @JsonProperty("server") Server
    }

    companion object {
      fun android(): Pair<Id, Map<Option.Id, Set<Variant.Id>>> = Id.Android to mapOf(Option.Id.UI to setOf(Variant.Id.Compose))
      fun ios(ui: Variant.Id): Pair<Id, Map<Option.Id, Set<Variant.Id>>> = Id.iOS to mapOf(Option.Id.UI to setOf(ui))
      fun desktop(): Pair<Id, Map<Option.Id, Set<Variant.Id>>> = Id.Desktop to mapOf(Option.Id.UI to setOf(Variant.Id.Compose))
      fun web(): Pair<Id, Map<Option.Id, Set<Variant.Id>>> = Id.Web to mapOf(Option.Id.UI to setOf(Variant.Id.Compose))
      fun server(): Pair<Id, Map<Option.Id, Set<Variant.Id>>> = Id.Server to mapOf(Option.Id.Engine to setOf(Variant.Id.Ktor))
    }
  }

  data class Option(@JsonProperty("variants") val variants: List<Variant>) {
    enum class Id {
      @JsonProperty("ui") UI,
      @JsonProperty("engine") Engine,
    }
  }

  data class Variant(@JsonProperty("id") val id: Id) {
    enum class Id {
      @JsonProperty(value = "xml") Xml,
      @JsonProperty("compose") Compose,
      @JsonProperty("swiftui") SwiftUI,
      @JsonProperty("ktor") Ktor,
    }
  }
}
