// Copyright 2000-2025 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
package git4idea.remoteApi

import com.intellij.dvcs.repo.Repository
import com.intellij.dvcs.repo.repositoryId
import com.intellij.openapi.components.service
import com.intellij.platform.vcs.impl.shared.rpc.FilePathDto
import com.intellij.vcs.git.ref.GitCurrentRef
import com.intellij.vcs.git.ref.GitFavoriteRefs
import com.intellij.vcs.git.repo.GitHash
import com.intellij.vcs.git.repo.GitOperationState
import com.intellij.vcs.git.rpc.GitRepositoryDto
import com.intellij.vcs.git.rpc.GitRepositoryStateDto
import com.intellij.vcsUtil.VcsUtil
import git4idea.GitStandardRemoteBranch
import git4idea.branch.GitBranchType
import git4idea.branch.GitTagType
import git4idea.repo.GitBranchTrackInfo
import git4idea.repo.GitRefUtil
import git4idea.repo.GitRepository
import git4idea.repo.tags
import git4idea.ui.branch.GitBranchManager
import org.jetbrains.annotations.VisibleForTesting

internal object GitRepositoryToDtoConverter {
  fun convertToDto(repository: GitRepository): GitRepositoryDto {
    return GitRepositoryDto(
      repositoryId = repository.repositoryId(),
      shortName = VcsUtil.getShortVcsRootName(repository.project, repository.root),
      state = convertRepositoryState(repository),
      favoriteRefs = collectFavorites(repository),
      root = FilePathDto.toDto(VcsUtil.getFilePath(repository.root)),
    )
  }

  fun convertRepositoryState(repository: GitRepository): GitRepositoryStateDto {
    val repoInfo = repository.info

    return GitRepositoryStateDto(
      currentRef = GitCurrentRef.wrap(GitRefUtil.getCurrentReference(repository)),
      revision = repository.currentRevision?.let { GitHash(it) },
      localBranches = repoInfo.localBranchesWithHashes.keys,
      remoteBranches = repoInfo.remoteBranchesWithHashes.keys.filterIsInstance<GitStandardRemoteBranch>().toSet(),
      tags = repository.tagsHolder.tags,
      workingTrees = repository.workingTreeHolder.getWorkingTrees(),
      recentBranches = repository.branches.recentCheckoutBranches,
      operationState = convertOperationState(repository),
      trackingInfo = convertTrackingInfo(repoInfo.branchTrackInfosMap)
    )
  }

  fun collectFavorites(repository: GitRepository): GitFavoriteRefs {
    val branchManager = repository.project.service<GitBranchManager>()
    return GitFavoriteRefs(
      localBranches = branchManager.getFavoriteRefs(GitBranchType.LOCAL, repository),
      remoteBranches = branchManager.getFavoriteRefs(GitBranchType.REMOTE, repository),
      tags = branchManager.getFavoriteRefs(GitTagType, repository),
    )
  }

  private fun convertOperationState(repository: GitRepository): GitOperationState = when (repository.state) {
    Repository.State.NORMAL -> GitOperationState.NORMAL
    Repository.State.MERGING -> GitOperationState.MERGE
    Repository.State.REBASING -> GitOperationState.REBASE
    Repository.State.GRAFTING -> GitOperationState.CHERRY_PICK
    Repository.State.REVERTING -> GitOperationState.REVERT
    Repository.State.DETACHED -> GitOperationState.DETACHED_HEAD
  }

  @VisibleForTesting
  fun convertTrackingInfo(trackingInfo: Map<String, GitBranchTrackInfo>): Map<String, GitStandardRemoteBranch> {
    val result = HashMap<String, GitStandardRemoteBranch>(trackingInfo.size)

    trackingInfo.forEach { (branchName, trackInfo) ->
      val remoteBranch = trackInfo.remoteBranch
      if (remoteBranch is GitStandardRemoteBranch) {
        result[branchName] = remoteBranch
      }
    }

    return result
  }
}