import type ts from "typescript/lib/tsserverlibrary"
import {
  areTypesMutuallyAssignableTsServer,
  getCompletionSymbolsTsServer,
  getElementTypeTsServer,
  getResolvedSignatureTsServer,
  getSymbolTypeTsServer,
  getTypePropertiesTsServer,
  getTypePropertyTsServer,
  getTypeTextTsServer,
} from "./get-element-type-ts-server"
import type {
  AreTypesMutuallyAssignableArguments,
  GetCompletionSymbolsArguments,
  GetElementTypeArguments,
  GetResolvedSignatureArguments,
  GetSymbolTypeArguments,
  GetTypePropertiesArguments,
  GetTypePropertyArguments,
  GetTypeTextArguments,
  Range,
  TestSleepArguments,
  TestSleepResponse
} from "./protocol"
import {
  areTypesMutuallyAssignable,
  getCompletionSymbols,
  getElementType,
  getResolvedSignature,
  getSymbolType,
  getTypeProperties,
  getTypeProperty,
  getTypeText,
  ReverseMapper
} from "./ide-get-element-type"
import {throwIdeError} from "./utils"

type TypeScript = typeof ts

type tsServerHandler<T> = (ts: TypeScript,
                           projectService: ts.server.ProjectService,
                           requestArgument: T) => ts.server.HandlerResponse | undefined

type lspHandler<T> = (ts: TypeScript,
                      requestArgument: T,
                      context: LspSupport) => Promise<ts.server.HandlerResponse | undefined>

interface IdeCommandArgumentsTypes {
  "ideGetCompletionSymbols": GetCompletionSymbolsArguments
  "ideGetElementType": GetElementTypeArguments
  "ideGetSymbolType": GetSymbolTypeArguments
  "ideGetTypeProperties": GetTypePropertiesArguments
  "ideGetTypeProperty": GetTypePropertyArguments,
  "ideGetTypeText": GetTypeTextArguments,
  "ideAreTypesMutuallyAssignable": AreTypesMutuallyAssignableArguments
  "ideGetResolvedSignature": GetResolvedSignatureArguments
  "ideCloseSafely": ts.server.protocol.FileRequestArgs
  "ideEnsureFileAndProjectOpenedCommand": ts.server.protocol.FileLocationRequestArgs,
  "ideTestSleep": TestSleepArguments,
}

const customHandlers: {
  [K in keyof IdeCommandArgumentsTypes]: [tsServerHandler<IdeCommandArgumentsTypes[K]>, lspHandler<IdeCommandArgumentsTypes[K]> | undefined]
} = {
  "ideGetCompletionSymbols": [getCompletionSymbolsTsServer, getCompletionSymbolsLsp],
  "ideGetElementType": [getElementTypeTsServer, getElementTypeLsp],
  "ideGetSymbolType": [getSymbolTypeTsServer, getSymbolTypeLsp],
  "ideGetTypeProperties": [getTypePropertiesTsServer, getTypePropertiesLsp],
  "ideGetTypeProperty": [getTypePropertyTsServer, getTypePropertyLsp],
  "ideGetTypeText": [getTypeTextTsServer, getTypeTextLsp],
  "ideAreTypesMutuallyAssignable": [areTypesMutuallyAssignableTsServer, areTypesMutuallyAssignableLsp],
  "ideGetResolvedSignature": [getResolvedSignatureTsServer, getResolvedSignatureLsp],
  "ideCloseSafely": [closeSafelyTsServer, undefined /* not supported on LSP */],
  "ideEnsureFileAndProjectOpenedCommand": [ensureFileAndProjectOpenedTsServer, undefined /* not supported on LSP */],
  "ideTestSleep": [testSleepTsServer, testSleepLsp], // Test-only command that puts the server to busy waiting
}

/** This method is used to register handlers for TS 5+ */
export function registerProtocolHandlers(
  session: ts.server.Session,
  ts: TypeScript,
  projectService: ts.server.ProjectService,
) {
  for (let command in customHandlers) {
    session.addProtocolHandler(command, (request: ts.server.protocol.Request) => {
      try {
        return customHandlers[command as keyof IdeCommandArgumentsTypes][0](ts, projectService, request.arguments) || emptyDoneResponse()
      }
      catch (e) {
        return processError(e)
      }
    });
  }
}

/** This method is used by the old session provider logic for TS <5 **/
export function initCommandNamesForSessionProvider(TypeScriptCommandNames: any) {
  for (let command in customHandlers) {
    TypeScriptCommandNames[command] = command
  }
}

/** This method is used by the old session provider logic for TS <5  **/
export function tryHandleTsServerCommand(ts_impl: TypeScript,
                                         projectService: ts.server.ProjectService,
                                         request: ts.server.protocol.Request): ts.server.HandlerResponse | undefined {
  try {
    return customHandlers[request.command as keyof IdeCommandArgumentsTypes]?.[0]?.(ts_impl, projectService, request.arguments)
  }
  catch (e) {
    return processError(e)
  }
}


export interface LspSupport {
  cancellationToken: ts.CancellationToken,

  process<T>(fileUri: string, range: Range | undefined, processor: (context: LspProcessingContext) => T): Promise<T | undefined>
}

export interface LspProcessingContext {
  cancellationToken: ts.CancellationToken,
  languageService: ts.LanguageService,
  program: ts.Program,
  sourceFile: ts.SourceFile,
  range: Range | undefined,
  reverseMapper: ReverseMapper
}

export async function tryHandleCustomTsServerCommandLsp(
  ts: TypeScript,
  commandName: any,
  requestArguments: any,
  context: LspSupport,
): Promise<ts.server.HandlerResponse | undefined> {
  try {
    return await customHandlers[commandName as keyof IdeCommandArgumentsTypes]?.[1]?.(ts, requestArguments, context)
  }
  catch (e) {
    return processError(e)
  }
}

function closeSafelyTsServer(
  _ts: TypeScript,
  projectService: ts.server.ProjectService,
  requestArguments: ts.server.protocol.FileRequestArgs,
) {
  projectService.ideProjectService.closeClientFileSafely(requestArguments.file)
  return notRequiredResponse()
}

function ensureFileAndProjectOpenedTsServer(
  _ts: TypeScript,
  projectService: ts.server.ProjectService,
  requestArguments: ts.server.protocol.FileLocationRequestArgs,
): ts.server.HandlerResponse {
  // Ensure that there is a project opened for the file
  projectService.ideProjectService.getProjectAndSourceFile(requestArguments.file, requestArguments.projectFileName)
  return emptyDoneResponse()
}

let lastIdeProjectId = 0

function getIdeProjectId(context: LspProcessingContext): number {
  return context.languageService.ideProjectId ?? (context.languageService.ideProjectId = lastIdeProjectId++)
}

function getCompletionSymbolsLsp(
  ts: TypeScript,
  requestArgument: GetCompletionSymbolsArguments,
  lspSupport: LspSupport
) {
  return lspSupport.process(requestArgument.file, undefined, context => {
    const {line, character} = requestArgument.position;
    let position = ts.getPositionOfLineAndCharacter(context.sourceFile, line, character);
    return getCompletionSymbols({
      ls: context.languageService,
      ts,
      program: context.program,
      sourceFileName: context.sourceFile.fileName,
      position,
      cancellationToken: context.cancellationToken,
      reverseMapper: context.reverseMapper,
    });
  });
}

function getElementTypeLsp(
  ts: TypeScript,
  requestArgument: GetElementTypeArguments,
  lspSupport: LspSupport,
) {
  return lspSupport.process(requestArgument.file, requestArgument.range, context => {
    const ideProjectId = getIdeProjectId(context)
    const range = context.range
    if (!range)
      return undefined

    // TODO consider using languageService.webStormGetElementType
    return getElementType({
      ts,
      ideProjectId,
      program: context.program,
      sourceFile: context.sourceFile,
      range,
      typeRequestKind: requestArgument.typeRequestKind,
      forceReturnType: requestArgument.forceReturnType,
      cancellationToken: context.cancellationToken,
      reverseMapper: context.reverseMapper,
    })
  })
}

function getSymbolTypeLsp(
  ts: TypeScript,
  requestArgument: GetSymbolTypeArguments,
  lspSupport: LspSupport,
) {
  return lspSupport.process(requestArgument.originalRequestUri, undefined, context => {
    if (context.languageService.ideProjectId !== requestArgument.ideProjectId
      || context.program.getTypeChecker().webStormCacheInfo?.ideTypeCheckerId !== requestArgument.ideTypeCheckerId) {
      return undefined;
    }
    return getSymbolType({
      ts,
      program: context.program,
      symbolId: requestArgument.symbolId,
      cancellationToken: context.cancellationToken,
      reverseMapper: context.reverseMapper,
    })
  })
}

function getTypePropertiesLsp(
  ts: TypeScript,
  requestArgument: GetTypePropertiesArguments,
  lspSupport: LspSupport,
) {
  return lspSupport.process(requestArgument.originalRequestUri, undefined, context => {
    if (!isCorrectProjectIdAndTypeCheckerId(context, requestArgument.ideProjectId, requestArgument.ideTypeCheckerId)) return undefined;
    return getTypeProperties({
      ts,
      program: context.program,
      typeId: requestArgument.typeId,
      cancellationToken: context.cancellationToken,
      reverseMapper: context.reverseMapper,
    })
  })
}

function getTypePropertyLsp(
  ts: TypeScript,
  requestArgument: GetTypePropertyArguments,
  lspSupport: LspSupport,
) {
  return lspSupport.process(requestArgument.originalRequestUri, undefined, context => {
    if (!isCorrectProjectIdAndTypeCheckerId(context, requestArgument.ideProjectId, requestArgument.ideTypeCheckerId)) return undefined;
    return getTypeProperty({
      ts,
      program: context.program,
      typeId: requestArgument.typeId,
      propertyName: requestArgument.propertyName,
      cancellationToken: context.cancellationToken,
      reverseMapper: context.reverseMapper,
    })
  })
}

function getTypeTextLsp(ts: TypeScript, requestArgument: GetTypeTextArguments,
                        lspSupport: LspSupport) {
  return lspSupport.process(requestArgument.originalRequestUri, undefined, context => {
    if (!isCorrectProjectIdAndTypeCheckerId(context, requestArgument.ideProjectId, requestArgument.ideTypeCheckerId)) return undefined;
    return getTypeText({
      ts,
      program: context.program,
      typeId: requestArgument.typeId,
      symbolId: requestArgument.symbolId,
      flags: requestArgument.flags,
    })
  })
}

function areTypesMutuallyAssignableLsp(
  ts: TypeScript,
  requestArgument: AreTypesMutuallyAssignableArguments,
  lspSupport: LspSupport,
) {
  return lspSupport.process(requestArgument.originalRequestUri, undefined, context => {
    if (!isCorrectProjectIdAndTypeCheckerId(context, requestArgument.ideProjectId, requestArgument.ideTypeCheckerId)) return undefined;
    return areTypesMutuallyAssignable({
      ts,
      program: context.program,
      type1Id: requestArgument.type1Id,
      type2Id: requestArgument.type2Id,
      cancellationToken: context.cancellationToken,
    })
  })
}

function getResolvedSignatureLsp(
  ts: TypeScript,
  requestArgument: GetResolvedSignatureArguments,
  lspSupport: LspSupport,
) {
  return lspSupport.process(requestArgument.file, requestArgument.range, context => {
    const ideProjectId = getIdeProjectId(context)
    if (!context.range) return undefined;
    return getResolvedSignature({
      ts,
      ideProjectId,
      program: context.program,
      sourceFile: context.sourceFile,
      range: context.range,
      cancellationToken: context.cancellationToken,
      reverseMapper: context.reverseMapper,
    })
  })
}

function isCorrectProjectIdAndTypeCheckerId(
  context: LspProcessingContext,
  ideProjectId: number,
  ideTypeCheckerId: number,
) {
  if (context.languageService.ideProjectId !== ideProjectId) return undefined;
  if (context.program.getTypeChecker().webStormCacheInfo?.ideTypeCheckerId !== ideTypeCheckerId) {
    throwIdeError("OutdatedTypeCheckerIdException")
  }
}

function testSleepTsServer(
  _ts: TypeScript,
  projectService: ts.server.ProjectService,
  _requestArguments: { durationMs: number, cancellable: boolean },
): TestSleepResponse {
  return testSleep(_requestArguments, projectService.cancellationToken)
}

function testSleep(
  requestArguments: TestSleepArguments,
  cancellationToken: ts.HostCancellationToken,
): TestSleepResponse {
  const start = Date.now()
  while (!requestArguments.cancellable || !cancellationToken.isCancellationRequested()) {
    if (Date.now() - start > requestArguments.durationMs) {
      return timeoutResponse()
    }
  }
  return cancelledResponse()
}

function processError(e: any): ts.server.HandlerResponse {
  let ideErrorKind = (e as Error).ideKind
  if (!ideErrorKind) throw e
  switch (ideErrorKind) {
    case "OperationCancelledException":
      return cancelledResponse()
    default:
      return {
        responseRequired: true,
        response: {
          error: ideErrorKind
        }
      }
  }
}

async function testSleepLsp(
  _ts: TypeScript,
  _requestArguments: TestSleepArguments,
  lspSupport: LspSupport,
): Promise<TestSleepResponse | undefined> {
  return testSleep(_requestArguments, lspSupport.cancellationToken)
}

function notRequiredResponse() {
  return {
    responseRequired: false
  }
}

function emptyDoneResponse(): { response: any; responseRequired: true } {
  return {
    responseRequired: true,
    response: null
  }
}

function cancelledResponse(): TestSleepResponse {
  return {
    responseRequired: true,
    response: {
      cancelled: true
    }
  }
}

function timeoutResponse(): TestSleepResponse {
  return {
    responseRequired: true,
    response: {
      cancelled: false
    }
  }
}
