// Copyright 2000-2023 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
package com.intellij.psi;

import com.intellij.ide.highlighter.HtmlFileType;
import com.intellij.ide.highlighter.XHtmlFileType;
import com.intellij.ide.highlighter.XmlFileType;
import com.intellij.lang.ASTFactory;
import com.intellij.lang.Language;
import com.intellij.lang.html.HTMLLanguage;
import com.intellij.lang.xml.XMLLanguage;
import com.intellij.openapi.diagnostic.Logger;
import com.intellij.openapi.fileTypes.FileType;
import com.intellij.openapi.fileTypes.LanguageFileType;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.html.HtmlTag;
import com.intellij.psi.util.PsiTreeUtil;
import com.intellij.psi.xml.XmlAttribute;
import com.intellij.psi.xml.XmlDocument;
import com.intellij.psi.xml.XmlElementType;
import com.intellij.psi.xml.XmlFile;
import com.intellij.psi.xml.XmlTag;
import com.intellij.psi.xml.XmlText;
import com.intellij.util.IncorrectOperationException;
import com.intellij.xml.util.XmlTagUtil;
import org.jetbrains.annotations.NonNls;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

/**
 * @author Dmitry Avdeev
 */
public class XmlElementFactoryImpl extends XmlElementFactory {

  private final Project myProject;

  public XmlElementFactoryImpl(Project project) {
    myProject = project;
  }

  @Override
  public @NotNull XmlTag createTagFromText(@NotNull @NonNls CharSequence text, @NotNull Language language) throws IncorrectOperationException {
    final FileType type = getFileType(language);
    final XmlDocument document = createXmlDocument(text, "dummy." + type.getDefaultExtension(), type);
    final XmlTag tag = document.getRootTag();
    if (tag == null) throw new IncorrectOperationException("Incorrect tag text");
    return tag;
  }

  @Override
  public @NotNull XmlTag createTagFromText(@NotNull CharSequence text) throws IncorrectOperationException {
    return createTagFromText(text, XMLLanguage.INSTANCE);
  }

  @Override
  public @NotNull XmlAttribute createXmlAttribute(@NotNull String name, @NotNull String value) throws IncorrectOperationException {
    return createAttribute(name, quoteValue(value), XmlFileType.INSTANCE);
  }

  @Override
  public @NotNull XmlAttribute createAttribute(@NotNull @NonNls String name, @NotNull String value, @Nullable PsiElement context)
    throws IncorrectOperationException {
    return createAttribute(name, quoteValue(value), getFileType(context));
  }

  @Override
  public @NotNull XmlAttribute createAttribute(@NotNull String name,
                                               @NotNull String value,
                                               @Nullable Character quoteStyle,
                                               @Nullable PsiElement context) throws IncorrectOperationException {
    return createAttribute(name, quoteValue(value, quoteStyle), getFileType(context));
  }

  private @NotNull XmlAttribute createAttribute(@NotNull String name, @NotNull String quotedValue, @NotNull FileType fileType) {
    final XmlDocument document = createXmlDocument("<tag " + name + "=" + quotedValue + "/>",
                                                   "dummy." + fileType.getDefaultExtension(), fileType);
    XmlTag tag = document.getRootTag();
    assert tag != null;
    XmlAttribute[] attributes = tag.getAttributes();
    LOG.assertTrue(attributes.length == 1, document.getText());
    return attributes[0];
  }

  public static @NotNull String quoteValue(@NotNull String value, @Nullable Character quoteStyle) {
    if (quoteStyle != null) {
      if (quoteStyle == '\'') {
        return quoteStyle + StringUtil.replace(value, "'", "&apos;") + quoteStyle;
      }
      else if (quoteStyle == '"') {
        return quoteStyle + StringUtil.replace(value, "\"", "&quot;") + quoteStyle;
      }
    }
    return quoteValue(value);
  }

  public static @NotNull String quoteValue(@NotNull String value) {
    final char quoteChar;
    if (!value.contains("\"")) {
      quoteChar = '"';
    }
    else if (!value.contains("'")) {
      quoteChar = '\'';
    }
    else {
      quoteChar = '"';
      value = StringUtil.replace(value, "\"", "&quot;");
    }
    return quoteChar + value + quoteChar;
  }

  @Override
  public @NotNull XmlText createDisplayText(@NotNull String s) throws IncorrectOperationException {
    final XmlTag tagFromText = createTagFromText("<a>" + XmlTagUtil.getCDATAQuote(s) + "</a>");
    final XmlText[] textElements = tagFromText.getValue().getTextElements();
    if (textElements.length == 0) return (XmlText)ASTFactory.composite(XmlElementType.XML_TEXT);
    return textElements[0];
  }

  @Override
  public @NotNull XmlTag createXHTMLTagFromText(@NotNull String text) throws IncorrectOperationException {
    final XmlDocument document = createXmlDocument(text, "dummy.xhtml", XHtmlFileType.INSTANCE);
    final XmlTag tag = document.getRootTag();
    assert tag != null : "No tag created from: " + text;
    return tag;
  }

  @Override
  public @NotNull XmlTag createHTMLTagFromText(@NotNull String text) throws IncorrectOperationException {
    final XmlDocument document = createXmlDocument(text, "dummy.html", HtmlFileType.INSTANCE);
    final XmlTag tag = document.getRootTag();
    assert tag != null : "No tag created from: " + text;
    return tag;
  }

  private XmlDocument createXmlDocument(final @NonNls CharSequence text, final @NonNls String fileName, FileType fileType) {
    PsiFile fileFromText = PsiFileFactory.getInstance(myProject).createFileFromText(fileName, fileType, text);

    XmlFile xmlFile;
    if (fileFromText instanceof XmlFile) {
      xmlFile = (XmlFile)fileFromText;
    }
    else {
      xmlFile = (XmlFile)fileFromText.getViewProvider().getPsi(((LanguageFileType)fileType).getLanguage());
      assert xmlFile != null;
    }
    XmlDocument document = xmlFile.getDocument();
    assert document != null;
    return document;
  }

  private static @NotNull FileType getFileType(@Nullable PsiElement context) {
    if (context == null) {
      return XmlFileType.INSTANCE;
    }
    if (context.getLanguage().isKindOf(HTMLLanguage.INSTANCE)) {
      return getFileType(context.getLanguage());
    }
    return PsiTreeUtil.getParentOfType(context, XmlTag.class, false) instanceof HtmlTag
           ? HtmlFileType.INSTANCE : XmlFileType.INSTANCE;
  }

  private static FileType getFileType(@NotNull Language language) {
    assert language instanceof XMLLanguage : "Tag can be created only for xml language";
    FileType type = language.getAssociatedFileType();
    return type == null ? XmlFileType.INSTANCE : type;
  }

  private static final Logger LOG = Logger.getInstance(XmlElementFactoryImpl.class);
}
